---
name: webextkits
description: |
  使用 WebExtKits 模板开发 Chrome 扩展时调用此 skill。
  涵盖：InjectScript 注入、background/inject 消息通信、
  chrome.storage.local 封装、Externals 配置等核心功能。
license: MIT
compatibility:
  mv3: true
  runtime:
    - chrome-extension
metadata:
  repo: FrontEndSharedProject/webextkits
---

# WebExtKits Chrome 扩展开发

WebExtKits 是一个 Chrome 扩展开发模板，提供完整的 TypeScript 支持和模块化架构。

## 核心概念

### 运行环境

| 环境 | 说明 | Chrome API |
|------|------|------------|
| Background | 扩展后台脚本 | ✅ 可用 |
| ContentScript | 内容脚本（隔离环境） | ⚠️ 部分可用（取决于 API） |
| InjectScript | 注入到页面的脚本 | ❌ 不可用 |
| Options/Popup | 扩展页面 | ✅ 可用 |

### ⚠️ 关键限制

**InjectScript 无法直接调用 Chrome API**

```typescript
// ❌ 错误：在 InjectScript 中直接调用
const data = await chrome.storage.local.get("user");

// ✅ 正确：通过消息传递给 Background
const data = await mc.send("readUserName");
```

InjectScript 运行在页面的 MAIN world 中，可以访问页面的 `window` 对象，但无法访问 Chrome 扩展 API。所有需要调用 Chrome API 的操作都必须通过消息传递给 Background 处理。

## 核心模块

### 1. 消息管理系统

用于 Background、InjectScript、Options、Popup 之间的通信。

```typescript
// InjectScript 端
import { MessagesInstance } from "@webextkits/messages-center/inject";
import type { InjectMessageType, BackgroundMessageType } from "@/messageType";

const mc = new MessagesInstance<InjectMessageType, BackgroundMessageType>(extId);
const userName = await mc.send("readUserName");
await mc.send("setUserName", userName);

// Background 端
import { MessageInstance } from "@webextkits/messages-center/background";
import type { InjectMessageType, BackgroundMessageType } from "@/messageType";

const mc = new MessageInstance<InjectMessageType, BackgroundMessageType>(extId);
mc.on("readUserName", async () => {
  const result = await chrome.storage.local.get("user");
  return result.user?.name ?? "";
});
mc.on("setUserName", async (name: string) => {
  await chrome.storage.local.set({ user: { name } });
});
```

→ 详见 [references/messages.md](references/messages.md)

### 2. 本地存储系统

对 `chrome.storage.local` 的类型安全封装。

```typescript
import { useStorageLocal } from "@webextkits/storage-local";
const { getBucket, setBucket, updateBucket } = useStorageLocal<SchemaType>(schema);

const user = await getBucket("user");
await updateBucket("user", (bucket) => {
  bucket.name = "新名称";
  return bucket;
});
```

→ 详见 [references/storage-local.md](references/storage-local.md)

### 3. InjectScript

注入到页面的脚本，可访问页面 `window` 对象。

```typescript
// 入口文件命名：*.entry.{ts,tsx}
// 示例：index.entry.tsx → dist/injects/index.js

// 在 Background 中注册
chrome.scripting.registerContentScripts([{
  id: "inject-index",
  js: ["injects/externals.js", "injects/index.js"],
  matches: ["https://example.com/*"],
  world: "MAIN",
}]);
```

→ 详见 [references/inject-script.md](references/inject-script.md)

### 4. Externals 配置

将第三方库提取到独立文件，大幅提升编译速度。

```typescript
// vite.config.ts
ViteWebExtKits({
  extensionId: extId,
  externals: ["react", "antd", "lodash-es"],
})
```

→ 详见 [references/externals.md](references/externals.md)

## 参考文档索引

| 文档 | 用途 |
|------|------|
| [project-structure.md](references/project-structure.md) | 项目目录结构、文件命名规则 |
| [messages.md](references/messages.md) | 消息通信系统完整 API |
| [storage-local.md](references/storage-local.md) | 本地存储 Schema 定义和使用 |
| [inject-script.md](references/inject-script.md) | InjectScript 注入和限制 |
| [externals.md](references/externals.md) | Externals 配置和优化 |
| [type-definitions.md](references/type-definitions.md) | 完整类型定义参考 |
| [best-practices.md](references/best-practices.md) | 最佳实践和故障排查 |

## 何时使用哪个文档

- **新建项目或添加新功能** → project-structure.md
- **实现 InjectScript 与 Background 通信** → messages.md
- **使用 chrome.storage.local** → storage-local.md
- **注入脚本到页面** → inject-script.md
- **优化编译速度** → externals.md
- **需要类型定义** → type-definitions.md
- **遇到问题** → best-practices.md
