# 最佳实践和故障排查

## 消息命名规范

```typescript
export type InjectMessageType = {
  // ✅ 推荐：动词 + 名词
  getUserProfile(): Promise<UserProfile>;
  setUserProfile(profile: UserProfile): Promise<void>;
  deleteUserData(): Promise<void>;
  fetchRemoteConfig(): Promise<Config>;

  // ❌ 避免：模糊的命名
  getData(): Promise<any>;
  update(data: any): Promise<void>;
  doSomething(): Promise<void>;
};
```

## Schema 组织

```typescript
// ✅ 推荐：按功能拆分
// schema/user.ts
export type UserSchemaType = { /* ... */ };

// schema/settings.ts
export type SettingsSchemaType = { /* ... */ };

// schema/cache.ts
export type CacheSchemaType = { /* ... */ };

// ❌ 避免：所有数据放在一个大 schema 中
export type AppSchemaType = {
  user: { /* ... */ },
  settings: { /* ... */ },
  cache: { /* ... */ },
};
```

## 错误处理

### InjectScript 端

```typescript
try {
  await mc.send("riskyOperation", data);
} catch (error) {
  const msg = error instanceof Error ? error.message : String(error);
  console.error("操作失败:", msg);
  // 显示用户友好的错误消息
}
```

### Background 端

```typescript
mc.on("riskyOperation", async (data) => {
  try {
    await doSomething(data);
  } catch (error) {
    // 抛出的错误会被 InjectScript 的 catch 捕获
    const msg = error instanceof Error ? error.message : String(error);
    throw new Error(`操作失败: ${msg}`);
  }
});
```

## 类型安全

```typescript
// ✅ 使用明确的类型
export type UserProfile = {
  id: string;
  name: string;
  email: string;
};

export type InjectMessageType = {
  getUserProfile(): Promise<UserProfile>;
  setUserProfile(profile: UserProfile): Promise<void>;
};

// ❌ 避免使用 any
export type InjectMessageType = {
  getData(): Promise<any>;
  setData(data: any): Promise<void>;
};
```

## 常见使用模式

### InjectScript 调用 Chrome API

```typescript
// 1. 定义消息类型
export type InjectMessageType = {
  getChromeStorage(key: string): Promise<any>;
  setChromeStorage(key: string, value: any): Promise<void>;
  openNewTab(url: string): Promise<void>;
};

// 2. InjectScript 发送消息
const data = await mc.send("getChromeStorage", "user");
await mc.send("setChromeStorage", "user", { name: "张三" });
await mc.send("openNewTab", "https://google.com");

// 3. Background 处理消息
mc.on("getChromeStorage", async (key: string) => {
  const result = await chrome.storage.local.get(key);
  return result[key];
});

mc.on("setChromeStorage", async (key: string, value: any) => {
  await chrome.storage.local.set({ [key]: value });
});

mc.on("openNewTab", async (url: string) => {
  await chrome.tabs.create({ url });
});
```

### 组合使用 Messages 和 StorageLocal

```typescript
// background/index.ts
import { schema, SchemaType } from "@/schema";
import { useStorageLocal } from "@webextkits/storage-local";

const storage = useStorageLocal<SchemaType>(schema);

mc.on("saveUserData", async (userData) => {
  await storage.setBucket("user", userData);
  await mc.send("userDataUpdated", userData);
});

mc.on("getUserData", async () => {
  return await storage.getBucket("user");
});
```

### 多入口 InjectScript

```typescript
// background/index.ts
const scripts = [
  {
    id: "github-enhancer",
    files: ["injects/github.js"],
    matches: ["https://github.com/*"],
  },
  {
    id: "youtube-controls",
    files: ["injects/youtube.js"],
    matches: ["https://www.youtube.com/*"],
  },
];

scripts.forEach(script => {
  chrome.scripting.registerContentScripts([{
    id: script.id,
    js: ["injects/externals.js", ...script.files],
    matches: script.matches,
    runAt: "document_end",
    world: "MAIN",
  }]);
});
```

## 故障排查

### InjectScript 无法接收消息

**问题**：`mc.on()` 没有触发

**解决方案**：
1. 确保调用了 `mc.connect()`
2. 检查 `extId` 是否一致
3. 确认 Background 中正确发送消息

### Storage 数据丢失

**问题**：读取 storage 时返回 undefined

**解决方案**：
1. 使用 `autoFillDefault: true` 选项
2. 检查 schema 的 `default` 值是否设置
3. 确认 schema 已正确注册

### Externals 未生效

**问题**：修改代码后编译速度仍然很慢

**解决方案**：
1. 确认 `externals.ts` 中已导出对应的库
2. 检查 vite.config.ts 中的 `externals` 配置
3. 确保 Background 中先注入 `externals.js`

### 入口文件未编译

**问题**：添加了 `*.entry.tsx` 文件但没有编译

**解决方案**：
1. 重启开发服务器
2. 确认文件命名符合 `*.entry.{ts,tsx}` 格式
3. 检查文件是否在 `/src/scopes/injects/` 目录下

### Chrome API 调用报错

**问题**：在 InjectScript 中调用 Chrome API 报错

**原因**：InjectScript 运行在 MAIN world，无法访问 Chrome API

**解决方案**：通过消息传递给 Background 处理

```typescript
// ❌ 错误
const data = await chrome.storage.local.get("user");

// ✅ 正确
const data = await mc.send("getChromeStorage", "user");
```

### 脚本未注入到页面

**问题**：InjectScript 没有运行

**解决方案**：
1. 检查 `matches` 配置是否匹配当前 URL
2. 确认 `world: "MAIN"` 已设置
3. 检查 `externals.js` 是否在 `js` 数组第一位
4. 查看 Chrome 扩展错误日志

### 消息超时

**问题**：`mc.send()` 长时间无响应

**解决方案**：
1. 检查 Background 中是否注册了对应的消息处理器
2. 确认处理器没有抛出未捕获的异常
3. 检查是否有死循环或长时间阻塞操作

## 调试技巧

### 开启 Debug 模式

```typescript
const mc = new MessagesInstance<InjectMessageType, BackgroundMessageType>(
  extId,
  true  // 开启 debug 模式
);
```

### 查看消息日志

Debug 模式下，所有消息收发都会打印到控制台。

### 检查已连接的 Tab

```typescript
// Background 中
console.log("已连接的 tabs:", mc.tabs);
```
