# Externals 配置

Externals 将第三方库（如 react、antd）从主代码中提取出来，单独打包成 `externals.js`。

## 优势

- ⚠️ **大幅提升编译速度**（从 3s 降到 10ms，提升 300 倍）
- 减少主代码体积
- 第三方库只编译一次

## 工作原理

**未配置 Externals**：

```javascript
// dist/injects/index.js (3MB+)
const React = /* react 源码 6.91kb */;
const Antd = /* antd 源码 3MB */;

function App() { /* 你的代码 */ }
```

**配置 Externals 后**：

```javascript
// dist/injects/externals.js (3MB+, 只编译一次)
window["extension_externals"]["react"] = /* react 源码 */;
window["extension_externals"]["antd"] = /* antd 源码 */;

// dist/injects/index.js (10KB, 每次修改都要编译)
((React, Antd) => {
  function App() { /* 你的代码 */ }
})(
  window["extension_externals"]["react"],
  window["extension_externals"]["antd"]
);
```

## 默认配置

已包含以下库：

```typescript
[
  "react",
  "react-dom",
  "react-dom/client",
  "react/jsx-runtime",
  "antd",
  "lodash-es",
  "dayjs",
  "react-use",
  "@ant-design/icons"
]
```

## 添加自定义 Externals

### 步骤 1: 配置 vite.config.ts

```typescript
// vite.config.ts
import { defineConfig } from "vite";
import { ViteWebExtKits } from "@webextkits/vite-plugins";

export default defineConfig({
  plugins: [
    ViteWebExtKits({
      extensionId: extId,
      externals: ["jquery", "axios"],  // 添加自定义库
    }),
  ],
});
```

### 步骤 2: 在 externals.ts 中导出

```typescript
// /src/scopes/injects/externals.ts
import * as antd from "antd";
import React from "react";
import ReactDOM from "react-dom/client";
import JSXRuntime from "react/jsx-runtime";
import jquery from "jquery";
import axios from "axios";

export default {
  // ⚠️ key 必须与包名完全一致
  "antd": antd,
  "react": React,
  "react-dom/client": ReactDOM,
  "react/jsx-runtime": JSXRuntime,
  "jquery": jquery,
  "axios": axios,
};
```

**⚠️ 重要**：导出对象的 key 必须与包名一致。

## Tree Shaking（可选）

如果只使用库的部分功能：

```typescript
// externals.ts
import { Button, Modal, Input } from "antd";

export default {
  "antd": { Button, Modal, Input },  // 只导出用到的组件
};
```

**权衡**：
- ✅ 减小文件体积
- ❌ 每次使用新组件都要手动添加

## 在 Background 中注册

```typescript
chrome.scripting.registerContentScripts([
  {
    id: "inject-module",
    // ⚠️ externals.js 必须在入口文件之前
    js: ["injects/externals.js", "injects/index.js"],
    matches: ["https://github.com/*"],
    runAt: "document_end",
    world: "MAIN",
  },
]);
```

## 完整配置示例

```typescript
// vite.config.ts
import { defineConfig } from "vite";
import { ViteWebExtKits } from "@webextkits/vite-plugins";
import { extId } from "./src/const";

export default defineConfig({
  plugins: [
    ViteWebExtKits({
      extensionId: extId,
      externals: [
        "react",
        "react-dom",
        "react-dom/client",
        "react/jsx-runtime",
        "antd",
        "lodash-es",
        "dayjs",
        "@ant-design/icons",
        // 自定义添加
        "axios",
        "jquery",
      ],
    }),
  ],
});
```

```typescript
// externals.ts
import * as antd from "antd";
import React from "react";
import ReactDOM from "react-dom/client";
import JSXRuntime from "react/jsx-runtime";
import * as lodashEs from "lodash-es";
import dayjs from "dayjs";
import * as antdIcons from "@ant-design/icons";
import axios from "axios";
import jquery from "jquery";

export default {
  "antd": antd,
  "react": React,
  "react-dom/client": ReactDOM,
  "react/jsx-runtime": JSXRuntime,
  "lodash-es": lodashEs,
  "dayjs": dayjs,
  "@ant-design/icons": antdIcons,
  "axios": axios,
  "jquery": jquery,
};
```

## 故障排查

### Externals 未生效

1. 确认 `externals.ts` 中已导出对应的库
2. 检查 vite.config.ts 中的 `externals` 配置
3. 确保 Background 中先注入 `externals.js`

### 编译速度仍然很慢

1. 检查是否有未配置的大型依赖
2. 确认 externals.ts 中的 key 与包名完全一致
3. 重启开发服务器
