# InjectScript

InjectScript 是通过 `chrome.scripting.registerContentScripts` API 注入到页面的脚本。

## 为什么使用 InjectScript

| 特性 | ContentScript | InjectScript |
|------|---------------|--------------|
| 运行环境 | Isolated World | MAIN World |
| 访问页面 window | ❌ | ✅ |
| 调用 Chrome API | ✅ | ❌ |
| 修改页面变量 | ❌ | ✅ |

**使用场景**：需要访问或修改页面的 `window` 对象时使用 InjectScript。

## ⚠️ 重要限制

**InjectScript 无法直接调用 Chrome API**

```typescript
// ❌ 错误：在 InjectScript 中直接调用
const data = await chrome.storage.local.get("user");  // 会报错

// ✅ 正确：通过消息传递给 Background
const data = await mc.send("getUserData");
```

在 Background 中处理：

```typescript
mc.on("getUserData", async () => {
  const result = await chrome.storage.local.get("user");
  return result;
});
```

## 文件命名规则

**入口文件**：必须遵循 `*.entry.{ts,tsx}` 命名格式

| 入口文件 | 编译输出 |
|---------|---------|
| `index.entry.tsx` | `dist/injects/index.js` |
| `main.entry.ts` | `dist/injects/main.js` |
| `addFriends.entry.ts` | `dist/injects/addFriends.js` |

**模块文件**：放在同名文件夹下

```text
📦injects
 ┣ 📂index
 ┃ ┣ 📜App.tsx           # index.entry.tsx 的依赖
 ┃ ┗ 📜utils.ts
 ┣ 📂main
 ┃ ┗ 📜helper.ts         # main.entry.ts 的依赖
 ┣ 📜externals.ts
 ┣ 📜index.entry.tsx     # ✅ 入口文件
 ┗ 📜main.entry.tsx      # ✅ 入口文件
```

**注意**：
- 只有 `*.entry.{ts,tsx}` 文件会被编译为独立入口
- 其他文件只能作为模块被入口文件引用
- 修改入口文件后如果没有重新编译，请重启项目

## 在 Background 中注册

```typescript
// background.ts
chrome.scripting.registerContentScripts([
  {
    id: "inject-index",
    js: ["injects/externals.js", "injects/index.js"],  // ⚠️ externals 在前
    css: ["injects/index.css"],  // 如果有样式
    matches: ["https://github.com/*"],
    runAt: "document_end",
    world: "MAIN",  // ⚠️ 必须是 MAIN
  },
  {
    id: "inject-main",
    js: ["injects/externals.js", "injects/main.js"],
    matches: ["https://example.com/*"],
    runAt: "document_start",
    world: "MAIN",
  }
]);
```

**关键配置**：
- `world: "MAIN"`：必须设置，否则无法访问页面 window
- `js` 数组中 `externals.js` 必须在入口文件之前

## 入口文件示例

### React 应用

```typescript
// index.entry.tsx
import { App } from "./index/App";
import ReactDOM from "react-dom/client";

const root = document.createElement("div");
root.id = "my-extension-root";
document.body.appendChild(root);

ReactDOM.createRoot(root).render(<App />);
```

### 纯 TypeScript

```typescript
// main.entry.ts
import { initializeFeature } from "./main/initialize";

console.log("Main script loaded");
initializeFeature();
```

## 多入口配置

```typescript
// background/index.ts
const scripts = [
  {
    id: "github-enhancer",
    files: ["injects/github.js"],
    matches: ["https://github.com/*"],
  },
  {
    id: "youtube-controls",
    files: ["injects/youtube.js"],
    matches: ["https://www.youtube.com/*"],
  },
];

scripts.forEach(script => {
  chrome.scripting.registerContentScripts([{
    id: script.id,
    js: ["injects/externals.js", ...script.files],
    matches: script.matches,
    runAt: "document_end",
    world: "MAIN",
  }]);
});
```

## 访问页面 Window

```typescript
// index.entry.tsx

// 读取页面变量
const pageData = window.__PAGE_DATA__;

// 修改页面变量
window.myExtensionConfig = {
  enabled: true,
  version: "1.0.0"
};

// 拦截页面函数
const originalFetch = window.fetch;
window.fetch = async (...args) => {
  console.log("拦截到 fetch:", args[0]);
  return originalFetch.apply(window, args);
};
```

## 故障排查

### 入口文件未编译

1. 重启开发服务器
2. 确认命名符合 `*.entry.{ts,tsx}` 格式
3. 检查文件在 `/src/scopes/injects/` 目录下

### 无法访问 Chrome API

这是正常的，InjectScript 运行在 MAIN world，无法访问 Chrome API。使用消息传递给 Background 处理。

### 脚本未注入

1. 检查 `matches` 配置是否正确
2. 确认 `world: "MAIN"` 已设置
3. 检查 `externals.js` 是否在 `js` 数组第一位
