# 消息管理系统

消息管理系统用于 Background、InjectScript、Options、Popup 之间的通信，提供完整的 TypeScript 支持。

## ⚠️ 核心限制

InjectScript 中**无法直接调用 Chrome API**，必须通过消息传递给 Background 处理。

## 消息类型定义

在 `/src/messageType.ts` 中定义消息类型：

```typescript
// InjectScript → Background 的消息
export type InjectMessageType = {
  readUserName(): Promise<string>;
  setUserName(name: string): Promise<void>;
  callChromeAPI(apiName: string, params: any): Promise<any>;
};

// Background → InjectScript 的消息
export type BackgroundMessageType = {
  updateLoginStatus(isLogin: boolean): Promise<void>;
  notifyDataChange(data: any): Promise<void>;
};
```

**规则**：
- 所有消息都是异步的，返回值必须是 `Promise<T>`
- `InjectMessageType`：injectScript → background
- `BackgroundMessageType`：background → injectScript

## InjectScript 中使用

```typescript
// /src/scopes/injects/index.entry.tsx
import { extId } from "@/const";
import { InjectMessageType, BackgroundMessageType } from "@/messageType";
import { MessagesInstance } from "@webextkits/messages-center/inject";

const mc = new MessagesInstance<InjectMessageType, BackgroundMessageType>(
  extId,  // 扩展唯一标识符
  true    // debug 模式
);

// 发送消息到 Background
async function handleSave(name: string) {
  try {
    await mc.send("setUserName", name);
    console.log("保存成功");
  } catch (error) {
    console.error("保存失败", error);
  }
}

// 接收来自 background 的消息
mc.on("updateLoginStatus", async (isLogin: boolean) => {
  console.log("登录状态更新:", isLogin);
});

// 主动建立连接
mc.connect();

// 断开连接
mc.disconnect();

// 销毁实例
mc.destroy();
```

## Background 中使用

```typescript
// /src/scopes/background/index.ts
import { extId } from "@/const";
import { InjectMessageType, BackgroundMessageType } from "@/messageType";
import { MessageInstance } from "@webextkits/messages-center/background";

const mc = new MessageInstance<InjectMessageType, BackgroundMessageType>(
  extId,
  true
);

// 监听来自 InjectScript 的消息
mc.on("readUserName", async function() {
  const tabId = this.sender?.tab?.id;
  console.log("来自 tab:", tabId);

  const result = await chrome.storage.local.get("user");
  return result.user.name;
});

mc.on("setUserName", async function(name: string) {
  await chrome.storage.local.set({ user: { name } });

  const tabId = this.sender?.tab?.id;
  if (tabId) {
    await mc.sendByTabId(tabId, "updateLoginStatus", true);
  }
});

// 向所有已连接的页面广播
await mc.send("updateLoginStatus", true);

// 向指定 tabId 发送
await mc.sendByTabId(123, "updateLoginStatus", false);

// 广播并处理每个页面的返回结果
mc.sendWithCallback(
  "updateLoginStatus",
  (result, sender) => {
    console.log("收到回复:", result, "来自:", sender.tab?.id);
  },
  true
);
```

## Options/Popup 中使用

```typescript
// /src/scopes/options/App.tsx
import { extId } from "@/const";
import { InjectMessageType, BackgroundMessageType } from "@/messageType";
import { MessageInstance } from "@webextkits/messages-center/background";

const mc = new MessageInstance<InjectMessageType, BackgroundMessageType>(
  extId,
  true
);

mc.on("readUserName", async () => {
  const user = await chrome.storage.local.get("user");
  return user.name;
});
```

**注意**：如果多个页面监听同一消息，只有第一个调用 `sendResponse()` 的会成功。

## API 参考

### MessagesInstance（InjectScript 端）

```typescript
class MessagesInstance<MessageType, BackgroundMessageType> {
  constructor(slug: string, isDebugger?: boolean);

  // 发送消息到 Background
  send<T extends keyof MessageType>(
    action: T,
    ...payload: Parameters<MessageType[T]>
  ): Promise<Awaited<ReturnType<MessageType[T]>>>;

  // 监听来自 Background 的消息
  on<T extends keyof BackgroundMessageType>(
    action: T,
    fn: (...args: Parameters<BackgroundMessageType[T]>) => void
  ): () => void;

  connect(): void;
  disconnect(): void;
  destroy(): void;
}
```

### MessageInstance（Background 端）

```typescript
class MessageInstance<MessageType, BackgroundMessageType> {
  constructor(slug: string, isDebugger?: boolean);

  // 监听来自 InjectScript 的消息
  on<T extends keyof MessageType>(
    action: T,
    fn: (...args: Parameters<MessageType[T]>) => ReturnType<MessageType[T]>
  ): () => void;

  // 向所有已连接页面广播
  send<T extends keyof BackgroundMessageType>(
    action: T,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<void>;

  // 向指定 tabId 发送
  sendByTabId<T extends keyof BackgroundMessageType>(
    tabId: number,
    action: T,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<Awaited<ReturnType<BackgroundMessageType[T]>>>;

  // 广播并处理返回结果
  sendWithCallback<T extends keyof BackgroundMessageType>(
    action: T,
    callback: (
      result: Awaited<ReturnType<BackgroundMessageType[T]>>,
      sender: chrome.runtime.MessageSender
    ) => void,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<void>;
}
```

## 常见模式：InjectScript 调用 Chrome API

```typescript
// 1. 定义消息类型
export type InjectMessageType = {
  getChromeStorage(key: string): Promise<any>;
  setChromeStorage(key: string, value: any): Promise<void>;
  openNewTab(url: string): Promise<void>;
};

// 2. InjectScript 发送消息
const data = await mc.send("getChromeStorage", "user");
await mc.send("setChromeStorage", "user", { name: "张三" });
await mc.send("openNewTab", "https://google.com");

// 3. Background 处理消息
mc.on("getChromeStorage", async (key: string) => {
  const result = await chrome.storage.local.get(key);
  return result[key];
});

mc.on("setChromeStorage", async (key: string, value: any) => {
  await chrome.storage.local.set({ [key]: value });
});

mc.on("openNewTab", async (url: string) => {
  await chrome.tabs.create({ url });
});
```
