# 本地存储系统

StorageLocal 是对 `chrome.storage.local` 的封装，提供：
- ✅ 完整的 TypeScript 支持
- ✅ 自动填充缺失字段（基于 JSON Schema）
- ✅ 内置锁机制，避免数据竞争
- ✅ 便捷的数据操作 API

## 定义 Schema

### 步骤 1: 创建 Schema 文件

```typescript
// /src/schema/user.ts
import { JSONSchemaType } from "@webextkits/storage-local";

// 1. 定义 TypeScript 类型
export type UserSchemaType = {
  name: string;
  age: number;
  email?: string;
};

// 2. 定义 JSON Schema（用于自动填充默认值）
export const UserSchema: JSONSchemaType<UserSchemaType> = {
  type: "object",
  properties: {
    name: {
      type: "string",
      default: "未命名用户",  // ⚠️ 必须设置默认值
    },
    age: {
      type: "number",
      default: 18,
    },
    email: {
      type: "string",
      default: "",
      nullable: true,  // 可选字段
    },
  },
  default: {},  // ⚠️ 对象本身也需要默认值
  required: ["name", "age"],
};
```

## Schema 编写建议

- **优先保证 default/required 完整**：尤其是 object 本身的 `default: {}` 以及必填字段列表，缺一很容易导致自动填充行为不符合预期。
- **可以用生成器辅助，但要人工校对**：如果团队有 JSON Schema 生成工具（从 TS 类型生成），生成后务必检查 `default`、`required`、以及可选字段的 `nullable`/默认值配置。

### 步骤 2: 注册 Schema

```typescript
// /src/schema/index.ts
import { UserSchemaType, UserSchema } from "./user";
import { SettingsSchemaType, SettingsSchema } from "./settings";

// 定义 SchemaType
export type SchemaType = {
  user: UserSchemaType;
  settings: SettingsSchemaType;
};

// 导出 schema 对象
export const schema = {
  user: UserSchema,
  settings: SettingsSchema,
};
```

**⚠️ 重要**：`SchemaType` 和 `schema` 中的 key 必须完全一致。

## 使用 StorageLocal

```typescript
import { schema, SchemaType } from "@/schema/index";
import { useStorageLocal } from "@webextkits/storage-local";

const { updateBucket, setBucket, getBucket, deleteBucket } =
  useStorageLocal<SchemaType>(schema);

// 1. 获取数据
const user = await getBucket("user");
console.log(user.name);

// 2. 更新数据
await updateBucket("user", (bucket) => {
  bucket.name = "新名称";
  bucket.age = 25;
  return bucket;  // ⚠️ 必须返回更新后的 bucket
});

// 3. 设置整个 bucket
await setBucket("user", {
  name: "张三",
  age: 30,
  email: "zhangsan@example.com"
});

// 4. 删除数据
await deleteBucket("user");

// 5. 自动填充默认值
const user = await getBucket("user", { autoFillDefault: true });
```

## 数据竞争保护

StorageLocal 内部使用 `navigator.locks` API：
- `getBucket`：shared 锁（允许并发读取）
- `updateBucket`：exclusive 锁（独占访问）
- `setBucket`：exclusive 锁
- `deleteBucket`：exclusive 锁

**无需手动处理锁**，系统自动管理。

## Reducer 用法

监听所有数据变更：

```typescript
const storage = useStorageLocal<SchemaType>(
  schema,
  (name, type, data) => {
    console.log(`${type} 操作:`, name, data);
  }
);
```

## API 参考

```typescript
type UseStorageLocalReturnType<BucketsType> = {
  updateBucket: <T extends keyof BucketsType>(
    name: T,
    callback: (payload: BucketsType[T]) => BucketsType[T] | Promise<BucketsType[T]>,
    options?: { autoFillDefault?: boolean }
  ) => Promise<void>;

  setBucket: <T extends keyof BucketsType>(
    name: T,
    bucket: BucketsType[T]
  ) => Promise<void>;

  getBucket: <T extends keyof BucketsType>(
    name: T,
    options?: { autoFillDefault?: boolean }
  ) => Promise<BucketsType[T]>;

  deleteBucket: <T extends keyof BucketsType>(
    name: T
  ) => Promise<void>;
};

function useStorageLocal<SchemaType>(
  schema: BucketsJSONSchemaType<SchemaType>,
  reducer?: (
    name: keyof SchemaType,
    type: "update" | "set" | "delete",
    data: SchemaType[keyof SchemaType]
  ) => void
): UseStorageLocalReturnType<SchemaType>;
```

## 与 Messages 组合使用

```typescript
// background/index.ts
import { schema, SchemaType } from "@/schema";
import { useStorageLocal } from "@webextkits/storage-local";

const storage = useStorageLocal<SchemaType>(schema);

mc.on("saveUserData", async (userData) => {
  await storage.setBucket("user", userData);
  await mc.send("userDataUpdated", userData);
});

mc.on("getUserData", async () => {
  return await storage.getBucket("user");
});
```
