# 完整类型定义

## @webextkits/messages-center

### 消息对象类型

```typescript
export type MessageObj<T> = {
  [K in keyof T]: (...args: any) => void;
};
```

### MessagesInstance（InjectScript 端）

```typescript
export class MessagesInstance<
  MessageType extends MessageObj<MessageType>,
  BackgroundMessageType extends MessageObj<BackgroundMessageType>
> {
  id: string;
  slug: string;
  isDebugger: boolean;
  listeners: Partial<Record<keyof BackgroundMessageType, any>>;

  constructor(slug: string, isDebugger?: boolean);

  send<T extends keyof MessageType>(
    action: T,
    ...payload: Parameters<MessageType[T]>
  ): Promise<Awaited<ReturnType<MessageType[T]>>>;

  on<T extends keyof BackgroundMessageType>(
    action: T,
    fn: (...args: Parameters<BackgroundMessageType[T]>) => void
  ): () => void;

  connect(): void;
  disconnect(): void;
  destroy(): void;
}
```

### MessageInstance（Background/Options/Popup 端）

```typescript
export class MessageInstance<
  MessageType extends MessageObj<MessageType>,
  BackgroundMessageType extends MessageObj<BackgroundMessageType>
> {
  slug: string;
  isDebugger: boolean;
  listeners: Partial<Record<keyof MessageType, any>>;
  tabs: number[];

  constructor(slug: string, isDebugger?: boolean);

  on<T extends keyof MessageType>(
    action: T,
    fn: (...args: Parameters<MessageType[T]>) => ReturnType<MessageType[T]>
  ): () => void;

  send<T extends keyof BackgroundMessageType>(
    action: T,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<void>;

  sendByTabId<T extends keyof BackgroundMessageType>(
    tabId: number,
    action: T,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<Awaited<ReturnType<BackgroundMessageType[T]>>>;

  sendWithCallback<T extends keyof BackgroundMessageType>(
    action: T,
    callback: (
      result: Awaited<ReturnType<BackgroundMessageType[T]>>,
      sender: chrome.runtime.MessageSender
    ) => void,
    ...payload: Parameters<BackgroundMessageType[T]>
  ): Promise<void>;
}
```

### 工具类型

```typescript
export type SendFn<T extends MessageObj<T>> = MessagesInstance<T, T>["send"];
export type OnFn<T extends MessageObj<T>> = MessagesInstance<T, T>["on"];
```

## @webextkits/storage-local

### JSON Schema 类型

```typescript
import type { JSONSchemaType } from "ajv";

export type BucketsJSONSchemaType<T extends Object> = {
  [K in keyof T]: JSONSchemaType<T[K]>;
};
```

### 通用选项

```typescript
type CommonOptions = {
  autoFillDefault?: boolean;
};
```

### useStorageLocal 返回类型

```typescript
type UseStorageLocalReturnType<BucketsType extends Object> = {
  updateBucket: <T extends keyof BucketsType>(
    name: T,
    callback: (
      payload: BucketsType[T]
    ) => BucketsType[T] | Promise<BucketsType[T]>,
    options?: CommonOptions
  ) => Promise<void>;

  setBucket: <T extends keyof BucketsType>(
    name: T,
    bucket: BucketsType[T]
  ) => Promise<void>;

  getBucket: <T extends keyof BucketsType>(
    name: T,
    options?: CommonOptions
  ) => Promise<BucketsType[T]>;

  deleteBucket: <T extends keyof BucketsType>(
    name: T
  ) => Promise<void>;
};
```

### useStorageLocal 函数

```typescript
export function useStorageLocal<SchemaType extends Object>(
  schema: BucketsJSONSchemaType<SchemaType>,
  reducer?: (
    name: keyof SchemaType,
    type: "update" | "set" | "delete",
    data: SchemaType[keyof SchemaType]
  ) => void
): UseStorageLocalReturnType<SchemaType>;
```

### JSON Schema 自动填充

```typescript
export function JSONSchemaAutoFill<T>(
  data: T | undefined,
  schema: JSONSchemaType<T>
): T;
```

### 导出 ajv 类型

```typescript
export type { JSONSchemaType } from "ajv";
```

## @webextkits/vite-plugins

### ViteWebExtKits 配置

```typescript
type ViteWebExtKitsOptions = {
  // 扩展 ID（必填）
  extensionId: string;

  // Externals 配置（可选）
  externals?: string[];

  // InjectScript 输入文件夹路径（可选）
  // 默认：./src/scopes/injects
  injectsInputFolderPath?: string;

  // Transformers 输入文件夹路径（可选）
  // 默认：./src/transformers
  transformersInputFolderPath?: string;
};

export function ViteWebExtKits(options: ViteWebExtKitsOptions): Plugin;
```

## 消息类型定义示例

```typescript
// /src/messageType.ts

// InjectScript → Background
export type InjectMessageType = {
  // 基本操作
  readUserName(): Promise<string>;
  setUserName(name: string): Promise<void>;

  // 带复杂参数
  callChromeAPI(apiName: string, params: any): Promise<any>;

  // 带对象参数
  saveUserData(data: {
    name: string;
    age: number;
    email?: string;
  }): Promise<void>;
};

// Background → InjectScript
export type BackgroundMessageType = {
  // 状态通知
  updateLoginStatus(isLogin: boolean): Promise<void>;

  // 数据更新通知
  notifyDataChange(data: {
    type: "user" | "settings";
    payload: any;
  }): Promise<void>;
};
```

## Schema 类型定义示例

```typescript
// /src/schema/user.ts
import { JSONSchemaType } from "@webextkits/storage-local";

export type UserSchemaType = {
  name: string;
  age: number;
  email?: string;
  preferences: {
    theme: "light" | "dark";
    notifications: boolean;
  };
};

export const UserSchema: JSONSchemaType<UserSchemaType> = {
  type: "object",
  properties: {
    name: { type: "string", default: "" },
    age: { type: "number", default: 0 },
    email: { type: "string", default: "", nullable: true },
    preferences: {
      type: "object",
      properties: {
        theme: {
          type: "string",
          enum: ["light", "dark"],
          default: "light",
        },
        notifications: { type: "boolean", default: true },
      },
      required: ["theme", "notifications"],
      default: { theme: "light", notifications: true },
    },
  },
  required: ["name", "age", "preferences"],
  default: {},
};
```

```typescript
// /src/schema/index.ts
import { UserSchemaType, UserSchema } from "./user";
import { SettingsSchemaType, SettingsSchema } from "./settings";

export type SchemaType = {
  user: UserSchemaType;
  settings: SettingsSchemaType;
};

export const schema = {
  user: UserSchema,
  settings: SettingsSchema,
};
```
